
// Gzip file access

#ifndef GZIP_FILE_H
#define GZIP_FILE_H

#include "abstract_file.h"

// Get size of gzipped file data (or size of file if not gzipped). NULL
// on success, otherwise error string.
const char* get_gzip_eof( const char* path, long* eof_out );


extern "C" {
class Data_Reader {
public:
	Data_Reader() { }
	virtual ~Data_Reader() { }
	
	// NULL on success, otherwise error string
	typedef const char* error_t;
	
	// Read at most 'n' bytes. Return number of bytes read, zero or negative
	// if error.
	virtual long read_avail( void*, long n ) = 0;
	
	// Read exactly 'n' bytes (error if fewer are available).
	virtual read( void*, long );
	
	// Number of bytes remaining
	virtual long remain() const = 0;
	
	// Skip forwards by 'n' bytes.
	virtual skip( long n );
	
	// to do: bytes remaining = LONG_MAX when unknown?
	
private:
	// noncopyable
	Data_Reader( const Data_Reader& );
	Data_Reader& operator = ( const Data_Reader& );
};

class File_Reader : public Data_Reader {
public:
	// Size of file
	virtual long size() const = 0;
	
	// Current position in file
	virtual long tell() const = 0;
	
	// Change position in file
	virtual  seek( long ) = 0;
	
	virtual long remain() const;
	
	skip( long n );
};
}

class Gzip_File_Reader : public File_Reader {
	void* file_;
	long size_;
public:
	Gzip_File_Reader();
	~Gzip_File_Reader();
	
	open( const char* );
	
	long size() const;
	long read_avail( void*, long );
	
	long tell() const;
	seek( long );
	
	void close();
};

class Gzip_File_Writer : public Data_Writer {
	void* file_;
public:
	Gzip_File_Writer();
	~Gzip_File_Writer();
	
	error_t open( const char* );
	error_t write( const void*, long );
	void close();
};

#endif

